/**
  *******************************************************************************
  * 
  * @file    hal_flash.c
  * @brief   ES7P203 flash module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The base driver can be used as follows:
    (+) Write flash by using flash_write().
    (+) Erase flash by using flash_erase()
    (+) Read flash by using flash_read()

  @endverbatim
  *********************************************************************************
  */
#include "hal_flash.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup flash base
  * @brief flash module driver
  * @{
  */

/**
  * @defgroup Flash_Public_Functions flash Public Functions
  * @brief    Flash Functions
@verbatim
=================================================================================
		##### Flash Public Functions #####
=================================================================================
  [..]  The flash driver can be used as follows:
    (+) Write flash by using flash_write().
    (+) Erase flash by using flash_erase()
    (+) Read flash by using flash_read()
@endverbatim
  * @{
  */

/**
  * @brief  Write flash
  * @param  addr: The address to write
  * @param  buf: The buffer to write data
  * @param  len: write size
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t flash_write(uint16_t addr, uint16_t *buf, uint16_t len)
{
	uint8_t  i = 0;
	uint16_t loop = 0;
	
	if (addr + len > FLASH_END)
		return ERROR;
	if (buf == NULL)
		return ERROR;
	
	assert_param(IS_FLASH_ADDR(addr));
	
	WDT_CLOSE();
	FLASH_UNLOCK();
	while (IAPUL == 0) {
		delay(100);
		i++;
		if (i == 100)
			return ERROR;
	}
	
	for (loop = 0; loop < len; loop++) {
		i = 0;
		FRAL   = addr;
		FRALN  = ~(addr & 0xff);
		FRAH   = (addr >> 8);
		FRAHN  = ~(addr >> 8);
		ROMDL  = buf[loop];
		ROMDLN = ~(buf[loop] & 0xff);
		ROMDH  = (buf[loop] >> 8);
		ROMDHN = ~(buf[loop] >> 8);
		FLASH_PROGRAMME();
		while (IAP_DONE != 1) {
			delay(100);
			i++;
			if (i == 100)
				return ERROR;
		}
		addr += 1;
	}
	
	FLASH_LOCK();
	return OK;
}

/**
  * @brief  Erase flash
  * @param  addr: The address to erase
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t flash_erase(uint16_t addr)
{
	uint8_t  i = 0;
	
	if (addr > FLASH_END)
		return ERROR;
	
	WDT_CLOSE();

	FRAL  = 0;
	FRALN = ~0;
	FRAH  = (addr >> 8);
	FRAHN = ~(addr >> 8);
	IAPCLR = 1;

	FLASH_UNLOCK();
	while (IAPUL == 0) {
		delay(100);
		i++;
		if (i == 100)
			return ERROR;
	}
	
	i = 0;
	FLASH_ERASE();
	while (IAP_DONE != 1) {
		delay(100);
		i++;
		if (i == 100)
			return ERROR;
	}

	FLASH_LOCK();
	return OK;
}

/**
  * @brief  Read flash
  * @param  ram_addr: The ram address store flash data
  * @param  addr: The address to flash 
  * @param  len: read size
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t flash_read(uint16_t *ram_addr, uint16_t addr, uint16_t len)
{
	uint16_t i;
	if (ram_addr == NULL)
		return ERROR;
	if (addr + len > FLASH_END)
		return ERROR;
	if (!len)
		return ERROR;

	assert_param(IS_FLASH_ADDR(addr));
	
	for (i = 0; i < len; i++) {
		ROMDH = 0;
		ROMDL = 0;

		FRAH  = addr >> 8;
		FRAHN = ~(addr >> 8);
		FRAL  = addr;
		FRALN = ~addr;
		__Asm TBR;
		*ram_addr = (ROMDH << 8) | ROMDL;
		 ram_addr++;
		 addr++;
	}

	return OK;
	
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
 